<?php
/* --------------------------------------------------------------
  ConfigWriter.inc.php 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

namespace StyleEdit\Repositories;

use \StyleEdit\JsonTransformer;
use \StyleEdit\ConfigSettings;
use \StyleEdit\Entities\StoredConfig;
use \StyleEdit\Entities\Config;
use \StyleEdit\Factories\ConfigFactory;

/**
 * Class ConfigWriter
 *
 * @package StyleEdit\Repositories
 */
class ConfigWriter
{
	/** @var \StyleEdit\ConfigSettings $settings */
	private $settings;
	
	/** @var \StyleEdit\Factories\ConfigFactory $configFactory */
	private $configFactory;
	
	
	/**
	 * ConfigWriter constructor.
	 *
	 * @param \StyleEdit\ConfigSettings          $settings
	 * @param \StyleEdit\Factories\ConfigFactory $configFactory
	 */
	public function __construct(ConfigSettings $settings, ConfigFactory $configFactory)
	{
		$this->settings      = $settings;
		$this->configFactory = $configFactory;
	}
	
	
	/**
	 * Updates the modification timestamp and writes the json encoded string into the file which is referenced by the
	 * style name.
	 *
	 * @param \StyleEdit\Entities\StoredConfig $styleConfig
	 * @param bool                             $p_toOriginalTheme Write files to original theme directory.
	 *
	 * @return \StyleEdit\Entities\StoredConfig
	 */
	public function update(StoredConfig $styleConfig, $p_toOriginalTheme = false)
	{
		// check if the style config name has changed and update its filename accordingly
		$this->_updateFilename($styleConfig);
		
		$filePath = $this->settings->getStylesDirectory($p_toOriginalTheme) . $styleConfig->getFilename();
		
		if(!$this->_isFileWritable($filePath))
		{
			throw new \RuntimeException('file "' . $filePath . '" is not writable');
		}
		
		$styleConfig->setModificationDate(new \DateTime());
		
		$jsonData = JsonTransformer::encode($styleConfig->getJsonDataArray());
		
		file_put_contents($filePath, $jsonData);
		
		return $styleConfig;
	}
	
	
	/**
	 * Creates a new style configuration by writing a file and returning a StoredConfig object representing
	 * the created style configuration file.
	 *
	 * @param \StyleEdit\Entities\Config $styleConfig
	 * @param bool                       $p_toOriginalTheme Write files to original theme directory.
	 *
	 * @return \StyleEdit\Entities\StoredConfig
	 */
	public function insert(Config $styleConfig, $p_toOriginalTheme = false)
	{
		$filename = $this->_generateFilename($styleConfig);
		$dirname  = $this->settings->getStylesDirectory($p_toOriginalTheme);
		$filePath = $dirname . $filename;
		$styleConfig->setCreationDate(new \DateTime());
		$styleConfig->setModificationDate(new \DateTime());
		
		$jsonData = JsonTransformer::encode($styleConfig->getJsonDataArray());
		
		// Ensure that child themes have the required "styleedit" directory. 
		if(!is_dir($dirname))
		{
			@mkdir($dirname, 0777, true);
			@chmod($dirname, 0777);
		}
		
		file_put_contents($filePath, $jsonData);
		
		$storedStyleConfig = $this->configFactory->createStoredConfig($filePath);
		
		return $storedStyleConfig;
	}
	
	
	/**
	 * Checks if the file specified by the given path is writable. Returns true or false.
	 *
	 * @param string $p_filePath A path to a file.
	 *
	 * @return bool
	 */
	private function _isFileWritable($p_filePath)
	{
		if(!is_writable($p_filePath))
		{
			// On some servers is_writable() returns false, even though the file is writable.
			if(@file_put_contents($p_filePath, file_get_contents($p_filePath)) === false)
			{
				return false;
			}
		}
		
		return true;
	}
	
	
	/**
	 * Creates and returns a filename based on a specific pattern and on the given configuration name.
	 * (e.i.: Style_Name_config_2015-12-06_18-06-52.json)
	 *
	 * @param \StyleEdit\Entities\Config $styleConfig
	 *
	 * @return string The generated filename.
	 */
	private function _generateFilename(Config $styleConfig)
	{
		$filename = $this->_generateFilenameStyleName($styleConfig) . '_config_' . date('Y-m-d_H-i-s') . '.json';
		
		return $filename;
	}
	
	
	/**
	 * Returns a filtered style name only allowing the characters a-z, A-Z, 0-9, _ and -
	 *
	 * @param \StyleEdit\Entities\Config $styleConfig
	 *
	 * @return mixed
	 */
	private function _generateFilenameStyleName(Config $styleConfig)
	{
		return preg_replace('/[^a-zA-Z0-9_-]/', '_', $styleConfig->getName());
	}
	
	
	/**
	 * Checks if the style config name has changed and updates its filename accordingly.
	 *
	 * @param \StyleEdit\Entities\StoredConfig $styleConfig
	 *
	 * @return \StyleEdit\Repositories\ConfigWriter Same instance to make chained method calls possible.
	 */
	private function _updateFilename(StoredConfig $styleConfig)
	{
		if(strpos($styleConfig->getFilename(), $this->_generateFilenameStyleName($styleConfig) . '_config_20') !== 0)
		{
			$filePath    = $this->settings->getStylesDirectory() . $styleConfig->getFilename();
			$newFileName = preg_replace('/.+?(_config_[\d]{4}-[\d]{2}-[\d]{2}_[\d]{2}-[\d]{2}-[\d]{2}\.json$)/',
			                            $this->_generateFilenameStyleName($styleConfig) . '$1',
			                            $styleConfig->getFilename());
			
			$newFilePath = $this->settings->getStylesDirectory() . $newFileName;
			
			$success = rename($filePath, $newFilePath);
			
			if($success)
			{
				$styleConfig->setFilename($newFileName);
			}
		}
		
		return $this;
	}
}